/* Copyright (c) 2005-2008 Di-an Jan.  All rights reserved.  */


import java.util.Iterator;			// 1.2
import java.util.SortedSet;			// 1.2
import java.util.TreeSet;			// 1.2

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Color;
import java.awt.Graphics;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.CardLayout;

import javax.swing.JComponent;			// 1.2
import javax.swing.JLabel;			// 1.2
import javax.swing.JToggleButton;		// 1.2
import javax.swing.JSpinner;			// 1.4

import javax.swing.JPanel;			// 1.2
import javax.swing.JApplet;			// 1.2

import javax.swing.BorderFactory;		// 1.2
import javax.swing.ButtonGroup;			// 1.2
import javax.swing.SpinnerNumberModel;		// 1.4

import java.awt.event.ComponentEvent;		// 1.1
import java.awt.event.ActionEvent;		// 1.1
import java.awt.event.MouseEvent;		// 1.1
import java.awt.event.ComponentListener;	// 1.1
import java.awt.event.ActionListener;		// 1.1
import java.awt.event.MouseListener;		// 1.1
import java.awt.event.MouseMotionListener;	// 1.1

import javax.swing.event.ChangeEvent;		// 1.2
import javax.swing.event.ChangeListener;	// 1.2



public final class FourierSeriesDemo extends JApplet {


    int		sign(int n)	{ return (n % 2 == 0) ? +1 : -1; }

    int		sq(int x)	{ return x * x; }
    double	sq(double x)	{ return x * x; }


static final int	DEFAULT_N_TERMS	= 5;
static final int	MAX_N_TERMS	= Integer.MAX_VALUE;
static final int	GRAPH_HEIGHT	= 240;
static final int	TIC		= 4;
static final int	CP_R		= 2;


abstract class Wave extends JToggleButton implements ActionListener
{
    abstract double	coef();
    abstract Term	term(int n);

    int			unit() { return 0; }

    Wave(String name)
    {
	super(name);
	addActionListener(this);
    }

    public void actionPerformed(ActionEvent event)
    {
	set_wave_type(this);
    }
}



final class EvenSquareWave extends Wave
{
    EvenSquareWave()	{ super("Even Square Wave"); }
    double coef()	{ return GRAPH_HEIGHT / Math.PI; }
    Term term(int n)	{ return Term.cos(2*n+1, sign(n), 2*n+1); }
}

final class OddSquareWave extends Wave
{
    OddSquareWave()	{ super("Odd Square Wave"); }
    double coef()	{ return GRAPH_HEIGHT / Math.PI; }
    Term term(int n)	{ return Term.sin(2*n+1, 1, 2*n+1); }
}

final class SawtoothWave1 extends Wave
{
    SawtoothWave1()	{ super("Sawtooth Wave 1"); }
    double coef()	{ return GRAPH_HEIGHT / (2 * Math.PI); }
    Term term(int n)	{ return Term.sin(n+1, -1, n+1); }
}

final class SawtoothWave0 extends Wave
{
    SawtoothWave0()	{ super("Sawtooth Wave 0"); }
    double coef()	{ return GRAPH_HEIGHT / (2 * Math.PI); }
    Term term(int n)	{ return Term.sin(n+1, sign(n), n+1); }
}

final class OddTriangleWave extends Wave
{
    OddTriangleWave()	{ super("Odd Triangle Wave"); }
    double coef()	{ return 2 * GRAPH_HEIGHT / sq(Math.PI); }
    Term term(int n)	{ return Term.sin(2*n+1, sign(n), sq(2*n+1)); }
}

final class EvenTriangleWave extends Wave
{
    EvenTriangleWave()	{ super("Even Triangle Wave"); }
    double coef()	{ return 2 * GRAPH_HEIGHT / sq(Math.PI); }
    Term term(int n)	{ return Term.cos(2*n+1, 1, sq(2*n+1)); }
}

final class ArbitraryWave extends Wave
{
    ArbitraryWave()	{ super("Arbitrary Function"); }
    double coef()	{ return 1; }
    int unit()		{ return 2; }
    Term term(int n)
    {
	if (n == 0)
	    return Term.cos(n, graph_panel.a_0());
	else if (n % 2 != 0)
	    return Term.cos((n+1)/2, graph_panel.a_n((n+1)/2));
	else // if (n % 2 == 0)
	    return Term.sin(n/2, graph_panel.b_n(n/2));
    }
}

final class EvenArbitraryWave extends Wave
{
    EvenArbitraryWave()	{ super("Arbitrary Even Function"); }
    double coef()	{ return 1; }
    int unit()		{ return 1; }
    Term term(int n)	{ return Term.cos(n, graph_panel.a_n(n)); }
}

final class OddArbitraryWave extends Wave
{
    OddArbitraryWave()	{ super("Arbitrary Odd Function"); }
    double coef()	{ return 1; }
    int unit()		{ return 1; }
    Term term(int n)	{ return Term.sin(n+1, graph_panel.b_n(n+1)); }
}


Wave	wave;
int	unit;

class TermsPanel extends JPanel
{
    TermsPanel()
    {
	setLayout(new WrapLayout());
	setPreferredSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
    }

    int  count()	{ return getComponentCount(); }
    void clear()	{ removeAll(); }
    Term get(int n)	{ return (Term) getComponent(n); }

    void set_n_terms(int n, boolean change_wave_type)
    {
	if (n <= count())
	{
	    for (int i = count() - 1; i >= n; i--)
		remove(i);
	    repaint(getVisibleRect());
	}
	else // n > count()
	{
	    for (int i = count(); i < n; i++)
		add(wave.term(i));
	    revalidate();
	    if (change_wave_type)
		repaint(getVisibleRect());
	}
	assert count() == n;
    }
} // final class TermsPanel






final class GraphPanel extends JPanel
			implements ComponentListener, MouseListener
{
    TreeSet<CP>	pts;
    int		width;
    int		height;
    int		unit;
    int		x0;
    int		x1;
    int		y0;
    int		y1;
    double	L;

    void redraw(Rectangle rect)
    {
	repaint(rect);
    }


    void redraw()
    {
	repaint(getVisibleRect());
    }


    final class CP extends JComponent implements Comparable<CP>, MouseMotionListener
    {
	int	x;
	int	y;
	int	pm;

	void set(int x, int y, int pm)
	{
	    this.x  = x;
	    this.y  = y;
	    this.pm = pm;
	    setBounds(x - CP_R, y - CP_R, 2 * CP_R + 1, 2 * CP_R + 1);
	    redraw();
	}

	CP(int x, int y, int pm)
	{
	    setBorder(BorderFactory.createLineBorder(Color.BLACK));
	    set(x, y, pm);
	    addMouseMotionListener(this);
	}

	CP succ()
	{
	    if (pm < 0)
		return new CP(x, y, +1);
	    else
		return new CP(x + 1, y, -1);
	}

	public int compareTo(CP r)
	{
		//	  - 0 +
		//	- 0 0 -
		//	0 0 0 0
		//	+ + 0 0

	    if (x < r.x)
		return -1;
	    if (x > r.x)
		return +1;
	    if (pm < 0 && r.pm > 0)
		return -1;
	    if (pm > 0 && r.pm < 0)
		return +1;

	    return 0;
	}

/*
	public int equals(Object obj)
	{
	    return x == r.x && (pm == r.pm || pm == 0 || r.pm == 0);
	}
*/

	public void mouseMoved(MouseEvent e) {}

	public void mouseDragged(MouseEvent e)
	{
	    assert unit != 0;

	    int x = getX() + e.getX();
	    int y = getY() + e.getY();

	    if (compareTo(pts.first()) == 0 || x < x0)
		x = x0;
	    if (compareTo(pts.last())  == 0 || x > x1)
		x = x1;

	    if (y < y1)
		y = y1;
	    if (y > y1 * 3)
		y = y1 * 3;

	    if (! pts.remove(this))
		assert false;

	    CP prev = prev_cp(this.x);
	    CP next = next_cp(this.x);

	    if (prev != null && x < prev.x)
		x = prev.x;
	    if (next != null && x > next.x)
		x = next.x;

	    if (prev != null && x == prev.x)
	    {
		if (y == prev.y || x == x0)
		{
		    pm = prev.pm;
		    del_cp(prev);
		}
		else
		{
		    if (prev.pm > 0)
			del_cp(prev);
		    else
			prev.pm = -1;
		    pm = +1;
		}
	    }
	    else if (next != null && x == next.x)
	    {
		if (y == next.y || x == x1)
		{
		    pm = next.pm;
		    del_cp(next);
		}
		else
		{
		    if (next.pm < 0)
			del_cp(next);
		    else
			next.pm = +1;
		    pm = -1;
		}
	    }
	    else
	    {
		if (prev != null && prev.pm < 0)
		    prev.pm = 0;
		if (next != null && next.pm > 0)
		    next.pm = 0;
		pm = 0;
	    }

	    set(x, y, pm);
	    pts.add(this);
	    set_wave_type(wave);
	}

	public String toString()
	{
	    if (pm > 0)
		return "(" + x + "+, " + y + ")";
	    if (pm < 0)
		return "(" + x + "-, " + y + ")";
	    else
		return "(" + x + " , " + y + ")";
	}

    } // class CP


    CP prev_cp(CP p)
    {
	SortedSet<CP> s = pts.headSet(p);
	if (s.isEmpty())
	    return null;
	return s.last();
    }

    CP next_cp(CP p)
    {
	SortedSet<CP> s = pts.tailSet(p.succ());
	if (s.isEmpty())
	    return null;
	return s.first();
    }

    CP prev_cp(int x)
    {
	return prev_cp(new CP(x, 0, +1));
    }

    CP next_cp(int x)
    {
	return next_cp(new CP(x, 0, -1));
    }

    void add_cp(CP p)
    {
	pts.add(p);
	add(p);
    }

    void del_cp(CP p)
    {
	pts.remove(p);
	remove(p);
    }


    GraphPanel(int unit)
    {
	this.unit = unit;
	setMinimumSize(new Dimension(0, GRAPH_HEIGHT));
	setPreferredSize(new Dimension(0, GRAPH_HEIGHT)); /* TODO: del */
	addComponentListener(this);

	if (unit != 0)
	{
	    setLayout(null);
	    pts = new TreeSet<CP>();
	    addMouseListener(this);
	}
    }


    public void mouseReleased(MouseEvent e)
    {
	assert unit != 0;

	int x = e.getX();
	int y = e.getY();

	if (y1 <= y && y <= y1 * 3 && x0 <= x && x <= x1)
	{
	    CP p = new CP(x, y, 0);
	    if (x == x0)
		p.pm = +1;
	    else if (x == x1)
		p.pm = -1;
	    if (! pts.contains(p))
	        add_cp(p);

	    set_wave_type(wave);
	}
    }

    public void mouseClicked(MouseEvent e) {}
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}
    public void componentShown(ComponentEvent e) {}
    public void componentHidden(ComponentEvent e) {}
    public void componentMoved(ComponentEvent e) {}

    public void componentResized(ComponentEvent e)
    {
	width  = getWidth();
	height = getHeight();	// GRAPH_HEIGHT;
	x0 = width / 2;
	x1 = x0 + unit * width / 8;
	L = width / 8.0;
	y0 = height / 2;
	y1 = height / 4;

	if (unit != 0 && getComponentCount() == 0 && width > 0)
	{
	    add_cp(new CP(x0, y0, 0));
	    add_cp(new CP(x1, y0, 0));
	}
    }

    protected void paintComponent(Graphics g)
    {
	super.paintComponent(g);

	if (unit != 0)
	{
	    CP a = null;
	    for (Iterator<CP> p = pts.iterator(); p.hasNext(); )
	    {
		CP b = p.next();
		if (a != null)
		    g.drawLine(a.x, a.y, b.x, b.y);
		a = b;
	    }
	}


	if (width <= 0)
	    // return;
	    throw new Error("Cannot paint before width is set.");

	g.drawLine(x0, 0, x0, height);

	for (int i = 1; i <= 3; i++) 
	    g.drawLine(0, i * height/ 4, width, i * height / 4);

	for (int i = 0; i <= 32; i++)
	    g.drawLine(i * width / (8 * 4), y0 - TIC,
	               i * width / (8 * 4), y0 + TIC);

	int a = 0;	// not needed
	for (int x = 0; x < width; x ++)
	{
	    double f = 0;

	    for (int n = 0; n < terms_panel.count(); n++)
		f += terms_panel.get(n).eval(x - 0.5 * width, L);
	    int b = (int)(0.5 * height - wave.coef() * f + 0.5);
	    if (x > 0)
		g.drawLine(x - 1, a, x, b);
	    a = b;
	}
    }


    double a_0()
    {
	double a_0 = 0;

	CP prev = null;
	for (Iterator<CP> p = pts.iterator(); p.hasNext(); )
	{
	    CP curr = p.next();
	    if (prev != null)
	    {
		if (prev.pm >= 0)
		    a_0 += (height - (curr.y + prev.y)) * (curr.x - prev.x) / (2 * L);
	    }
	    prev = curr;
	}
	if (unit == 1)
	    return 2 * a_0;

	return a_0;
    }


    double a_n(int n)
    {
	assert n >= 0;

	if (n == 0)
	    return a_0();

	double a_n = 0;
	double prev_m = 0;	// not needed
	double init_m = 0;	// not needed
	double init_y = 0;	// not needed
	boolean has_prev_m = false;

	CP prev = null;
	for (Iterator<CP> p = pts.iterator(); p.hasNext(); )
	{
	    CP curr = p.next();
	    if (prev != null)
	    {
		if (prev.pm >= 0)
		{
		    double curr_m = -(double)(curr.y - prev.y) / (curr.x - prev.x);
		    if (has_prev_m)
		    {
			a_n -= L * (curr_m - prev_m) * Math.cos(n*Math.PI*prev.x/L)
					/ sq(n * Math.PI);
		    }
		    else
			init_m = curr_m;

		    prev_m = curr_m;
		    has_prev_m = true;
		}
		else // if (prev.pm < 0)	// discontinuity
		{
		    a_n -= -(curr.y - prev.y) * Math.sin(n*Math.PI*prev.x/L)
				/ (n * Math.PI);
		}
	    }
	    else
		init_y = curr.y;

	    prev = curr;

	    if (! p.hasNext())
	    {
		if (unit == 2)
		{
		    a_n -= L * (init_m - prev_m) * Math.cos(n*Math.PI*prev.x/L)
				/ sq(n * Math.PI);
		    a_n -= -(init_y - prev.y) * Math.sin(n*Math.PI*prev.x/L)
				/ (n * Math.PI);
		}
		else
		{
		    a_n -= L * (init_m)
				/ sq(n * Math.PI);
		    a_n += L * (prev_m) * sign(n)
				/ sq(n * Math.PI);
		}
	    }
	}

	if (unit == 1)
	    return 2 * a_n;

	return a_n;
    }


    double b_n(int n)
    {
	assert n > 0;

	double b_n = 0;
	double prev_m = 0;	// not needed
	double init_m = 0;	// not needed
	double init_y = 0;	// not needed
	boolean has_prev_m = false;

	CP prev = null;
	for (Iterator<CP> p = pts.iterator(); p.hasNext(); )
	{
	    CP curr = p.next();
	    if (prev != null)
	    {
		if (prev.pm >= 0)
		{
		    double curr_m = -(double)(curr.y - prev.y) / (curr.x - prev.x);
		    if (has_prev_m)
		    {
			b_n -= L * (curr_m - prev_m) * Math.sin(n*Math.PI*prev.x/L)
					/ sq(n * Math.PI);
		    }
		    else
			init_m = curr_m;

		    prev_m = curr_m;
		    has_prev_m = true;
		}
		else // if (prev.pm < 0)	// discontinuity
		{
		    b_n += -(curr.y - prev.y) * Math.cos(n*Math.PI*prev.x/L)
				/ (n * Math.PI);
		}
	    }
	    else
		init_y = curr.y;

	    prev = curr;

	    if (! p.hasNext())
	    {
		if (unit == 2)
		{
		    b_n -= L * (init_m - prev_m) * Math.sin(n*Math.PI*prev.x/L)
				/ sq(n * Math.PI);
		    b_n += -(init_y - prev.y) * Math.cos(n*Math.PI*prev.x/L)
				/ (n * Math.PI);
		}
		else
		{
		    b_n -= (init_y - 0.5 * height)
				/ (n * Math.PI);
		    b_n += (prev.y - 0.5 * height) * sign(n)
				/ (n * Math.PI);
		}
	    }
	}

	if (unit == 1)
	    return 2 * b_n;

	return b_n;
    }


} // final class GraphPanel



public String getAppletInfo()
{
    return "Fourier Series Demo";
}

String[]	unit_names = { "0", "1", "2" };
GraphPanel[]	graph_panel_units;
GraphPanel	graph_panel;
JPanel		graph_panel_stack;
CardLayout	graph_panel_selector;

TermsPanel	terms_panel;

void set_n_terms(int n, boolean change_wave_type)
{
    terms_panel.set_n_terms(n, change_wave_type);
    graph_panel.redraw();
}


void set_wave_type(Wave w)
{
    int n_terms = terms_panel.count();
    if (n_terms == 0)
	n_terms = DEFAULT_N_TERMS;
    terms_panel.clear();
    assert terms_panel.count() == 0;
    wave = w;
    if (unit != wave.unit())
    {
	unit = wave.unit();
	graph_panel_selector.show(graph_panel_stack, unit_names[unit]);
    }
    graph_panel = graph_panel_units[unit];
    set_n_terms(n_terms, true);
}


Wave new_wave_button(Wave button, ButtonGroup button_group, boolean selected)
{
    button_group.add(button);
    if (selected)
	button_group.setSelected(button.getModel(), true);
    return button;
}

public FourierSeriesDemo()
{
    rootPane.setLayout(new BorderLayout());

    // graph panel and terms panels

    terms_panel = new TermsPanel();
    rootPane.add(terms_panel);

    graph_panel_selector = new CardLayout();
    graph_panel_stack = new JPanel(graph_panel_selector);
    graph_panel_units = new GraphPanel[3];
    for (int i = 0; i <= 2; i++)
    {
	graph_panel_units[i] = new GraphPanel(i);
	graph_panel_stack.add(graph_panel_units[i], unit_names[i]);
    }
    graph_panel = graph_panel_units[0];
    rootPane.add(graph_panel_stack, BorderLayout.NORTH);

    // control panel

    JPanel control_panel = new JPanel(new GridLayout(0, 1));

    control_panel.add(new JLabel("Fourier Series Demo"));

    Wave default_wave = new OddSquareWave();
    set_wave_type(default_wave);
    ButtonGroup button_group = new ButtonGroup();
    control_panel.add(new_wave_button(        default_wave,    button_group, true));
    control_panel.add(new_wave_button(new EvenSquareWave(),    button_group, false));
    control_panel.add(new_wave_button(new     SawtoothWave1(), button_group, false));
    control_panel.add(new_wave_button(new     SawtoothWave0(), button_group, false));
    control_panel.add(new_wave_button(new EvenTriangleWave(),  button_group, false));
    control_panel.add(new_wave_button(new  OddTriangleWave(),  button_group, false));
    control_panel.add(new_wave_button(new     ArbitraryWave(), button_group, false));
    control_panel.add(new_wave_button(new EvenArbitraryWave(), button_group, false));
    control_panel.add(new_wave_button(new  OddArbitraryWave(), button_group, false));

    JLabel label = new JLabel("n terms:");
    control_panel.add(label);
    JSpinner n_terms_control = new JSpinner(new
		SpinnerNumberModel(DEFAULT_N_TERMS, 1, MAX_N_TERMS, 1));
    n_terms_control.addChangeListener(new ChangeListener()
	{
	    public void stateChanged(ChangeEvent event)
	    {
		JSpinner n_terms_control = (JSpinner) event.getSource();
	        set_n_terms(((Integer) n_terms_control.getValue()).intValue(), false);
	    }
	});
    control_panel.add(n_terms_control);
    label.setLabelFor(n_terms_control);

    JPanel control_panel_panel = new JPanel(new BorderLayout());
    control_panel_panel.add(control_panel, BorderLayout.NORTH);
    rootPane.add(control_panel_panel, BorderLayout.WEST);
}


} // public final class FourierSeriesDemo

